package org.msh.tb.export_rest.core;

import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.streaming.SXSSFCell;
import org.apache.poi.xssf.streaming.SXSSFRow;
import org.apache.poi.xssf.streaming.SXSSFSheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.List;

/**
 * Service to export data from a reader to an Excel file
 * Created by rmemoria on 9/7/17.
 */
@Name("excelExportService")
@Scope(ScopeType.APPLICATION)
@AutoCreate
public class ExcelExportService {

    /**
     * Export data to an excel file from a single reader
     *
     * @param file
     * @param masterReader
     * @param childReaders
     * @param listener
     * @throws IOException
     */
    public boolean export(File file, BlockReader masterReader,
                          List<BlockReader> childReaders, ExcelExportListener listener) throws IOException {

        // create a new workbook
        SXSSFWorkbook workbook = new SXSSFWorkbook(100);

        // write the content of the workbook (and its sheets)
        if (!writeContent(workbook, masterReader, childReaders, listener)) {
            return false;
        }

        // generate the final excel file
        generateExcelFile(file, workbook);

        return true;
    }

    /**
     * Write the excel content to the given file
     *
     * @param file
     * @param workbook
     * @throws IOException
     */
    private void generateExcelFile(File file, SXSSFWorkbook workbook) throws IOException {
        FileOutputStream out = new FileOutputStream(file);
        try {
            workbook.write(out);
        } finally {
            out.close();
        }
    }

    /**
     * Write the titles as the first row in the given sheet
     *
     * @param sheet
     * @param masterReader
     * @param childReaders
     */
    private void writeTitles(SXSSFSheet sheet, BlockReader masterReader, List<BlockReader> childReaders) {
        SXSSFRow row = sheet.createRow(0);

        SXSSFCell cell = row.createCell(0);
        cell.setCellType(CellType.STRING);
        cell.setCellValue("Key");

        int colIndex = 1;

        addTitleColumns(row, masterReader.getColumns(), colIndex);
        colIndex += masterReader.getColumns().size();

        if (childReaders != null) {
            for (BlockReader reader : childReaders) {
                addTitleColumns(row, reader.getColumns(), colIndex);
                colIndex += reader.getColumns().size();
            }
        }
    }

    private void addTitleColumns(SXSSFRow row, List<String> titles, int colIndex) {
        for (String title : titles) {
            SXSSFCell cell = row.createCell(colIndex++);
            cell.setCellType(CellType.STRING);
            cell.setCellValue(title);
        }
    }


    /**
     * Write the content of the excel file, reading all the records and filling in
     * the spreadsheet
     *
     * @param workbook
     * @param masterReader
     * @param childReader
     * @param listener
     * @return
     */
    private boolean writeContent(SXSSFWorkbook workbook, BlockReader masterReader,
                                 List<BlockReader> childReader,
                                 ExcelExportListener listener) {
        int sheetIndex = 1;
        SXSSFSheet sheet = null;

        // first row is for the titles
        int rowIndex = 1;
        long recordIndex = 0;
        long recordCount = masterReader.getRecordCount();

        CellStyle dateStyle = createDateStyle(workbook);

        while (masterReader.next() != null) {

            if (sheet == null) {
                // create a new sheet
                sheet = workbook.createSheet("Data " + sheetIndex);
                // write the titles
                writeTitles(sheet, masterReader, childReader);
            }

            // create a new row
            SXSSFRow row = sheet.createRow(rowIndex);

            // fill the row
            writeRowContent(row, masterReader, childReader, dateStyle);

            // increment counters
            rowIndex++;
            recordIndex++;

            // check if it has reached the limit of the sheet size
            if (rowIndex > 1000000) {
                rowIndex = 1;
                sheet = null;
                sheetIndex++;
            }

            // notify the progress of the exporting
            if (!notifyProgress(listener, recordIndex, recordCount)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Create the date style for cells with date
     *
     * @param wb
     * @return
     */
    private CellStyle createDateStyle(Workbook wb) {
        CreationHelper createHelper = wb.getCreationHelper();
        CellStyle cellStyle = wb.createCellStyle();
        cellStyle.setDataFormat(createHelper.createDataFormat().getFormat("dd/mm/yyyy"));
        return cellStyle;
    }

    /**
     * Notify to the caller of the service about the progress of the exporting
     *
     * @param listener the listener to be informed
     * @param index    the index of the exporting
     * @param count    the total number of records to be imported
     * @return true if the exporting must continue, or false if the exporting was canceled
     */
    private boolean notifyProgress(ExcelExportListener listener, long index, long count) {
        if (listener == null) {
            return true;
        }

        double pos = count > 0 ? (double) index / (double) count : 0;
        return listener.onProgress(pos);
    }

    private void writeRowContent(SXSSFRow row, BlockReader masterReader, List<BlockReader> childReaders, CellStyle dateStyle) {
        Record rec = masterReader.current();
        Object key = rec.getKey();

        // write key
        writeCell(row, 0, key, dateStyle);

        // write master block row
        int colIndex = 1;
        for (Object val : rec.getValues()) {
            writeCell(row, colIndex, val, dateStyle);
            colIndex++;
        }

        if (childReaders == null) {
            return;
        }

        // write the content of the readers in the row by the row key
        for (BlockReader reader : childReaders) {
            // get the current record in the reader (or the first)
            rec = reader.current();

            // check if the current record has the same key as the master record
            // if the key is not the same, it is expected that the child record will be used
            // more ahead in the loop
            if (rec != null && rec.getKey().equals(key)) {
                int index = colIndex;
                // write content of the child reader
                for (Object val : rec.getValues()) {
                    writeCell(row, index, val, dateStyle);
                    index++;
                }

                // move to the next record in order to point the child reader to the next key
                // and be compared in the next iteration
                reader.next();
            }
            colIndex += reader.getColumns().size();
        }
    }

    /**
     * Write a cell value in the row
     *
     * @param row
     * @param colIndex
     * @param value
     * @return
     */
    private SXSSFCell writeCell(SXSSFRow row, int colIndex, Object value, CellStyle dateStyle) {
        SXSSFCell cell = row.createCell(colIndex);

        if (value == null) {
            return cell;
        }

        if (value instanceof Date) {
            writeDateCell(cell, (Date) value, dateStyle);
            return cell;
        }

        if (value instanceof Number) {
            writeNumericCell(cell, (Number) value);
            return cell;
        }

        if (value instanceof Boolean) {
            writeBooleanCell(cell, (Boolean) value);
            return cell;
        }

        cell.setCellValue(value.toString());
        cell.setCellType(CellType.STRING);
        return cell;
    }


    /**
     * Write a boolean value to a cell
     *
     * @param cell
     * @param val
     */
    private void writeBooleanCell(SXSSFCell cell, boolean val) {
        cell.setCellType(CellType.BOOLEAN);
        cell.setCellValue(val);
    }

    /**
     * Write a numeric value to a cell
     *
     * @param cell
     * @param val
     */
    private void writeNumericCell(SXSSFCell cell, Number val) {
        cell.setCellType(CellType.NUMERIC);
        cell.setCellValue(val.doubleValue());
    }

    /**
     * Write a date value in a cell
     *
     * @param cell
     * @param value
     */
    private void writeDateCell(SXSSFCell cell, Date value, CellStyle dateStyle) {
        cell.setCellValue(value);
        cell.setCellStyle(dateStyle);
    }


    /**
     * Calculate the sheet number by the record row to import
     *
     * @param recordIndex the
     * @return
     */
    private int calcSheetByRecord(long recordIndex) {
        return (int) (recordIndex / 1000000L);
    }

    /**
     * Return the sheet by row number
     *
     * @param workbook
     * @param recordIndex
     * @return
     */
    private WritableSheet sheetByRowNumber(WritableWorkbook workbook, long recordIndex) {
        int index = calcSheetByRecord(recordIndex);

        // check if the number of sheets are ok
        while (index >= workbook.getNumberOfSheets()) {
            int sheetNum = workbook.getNumberOfSheets();
            workbook.createSheet("Data " + sheetNum, sheetNum - 1);
        }

        return workbook.getSheet(index);
    }
}
